#ifndef _lf_switch_h_
#define _lf_switch_h_

#include <stdio.h>

/*
 * Structure describing one xbar
 */
struct lf_xbar_data {

  /* current values */
  struct lf_xbar_metrics {
    unsigned int quadrant_disable;
  } cur_vals;

   /* old values for computing deltas */
  struct lf_xbar_metrics old_vals;

   /* new values being read (possibly inconsistant) */
  struct lf_xbar_metrics new_vals;
};

/*
 * Structure describing one xbar port
 */
struct lf_xbarport_data {

  int xbarportno;	/* XbarPort number from monitoring card */

  /* current values */
  struct lf_xbarport_metrics {
    uint64_t goodcrcs;
    unsigned int badcrcs;
    unsigned int invalidroutes;
    unsigned int receivetimeoutcount;
    unsigned int transmittimeoutcount;
    unsigned int portdown;
    unsigned int portflipcount;
    unsigned int control;
  } cur_vals;

   /* old values for computing deltas */
  struct lf_xbarport_metrics old_vals;

   /* new values being read (possibly inconsistant) */
  struct lf_xbarport_metrics new_vals;
};

/*
 * Structure describing one xcvr
 */
struct lf_xcvr_data {

  /* current values */
  struct lf_xcvr_metrics {
    unsigned int signallost;
    unsigned int signallostcount;
    unsigned int control;
  } cur_vals;

   /* old values for computing deltas */
  struct lf_xcvr_metrics old_vals;

   /* new values being read (possibly inconsistant) */
  struct lf_xcvr_metrics new_vals;
};

/*
 * Descriptor for each slot
 */
struct lf_linecard_data {
  int seen;			/* used to notice missing linecards */
  int not_seen_cnt;
  int change_cnt;

#define LF_SWITCH_NUM_TEMPS 2

  struct lf_linecard_metrics {
    int temperature[LF_SWITCH_NUM_TEMPS];	/* card temperatures */
    int overtempcount;
  } cur_vals;

   /* old values for computing deltas */
  struct lf_linecard_metrics old_vals;

   /* new values being read (possibly inconsistant) */
  struct lf_linecard_metrics new_vals;
};

/*
 * There are different styles of output from the switches
 */
enum {
  LF_SWITCH_PROTO_UNKNOWN = 0,
  LF_SWITCH_PROTO_X16_HTML,
  LF_SWITCH_PROTO_X32_HTML,
  LF_SWITCH_PROTO_TELNET_1
};

/*
 * structure defining an entire enclosure.
 */
struct lf_enclosure_data {
  int data_proto;		/* what data protocol to use */

  struct lf_switch_data_change *change_list;	/* changes from baseline */
};

/*
 * A change which has occurred in a switch
 */
enum {
  LF_SWITCH_CHANGE_MISSING_LINECARD,
  LF_SWITCH_CHANGE_NEW_LINECARD,
  LF_SWITCH_CHANGE_CHANGED_LINECARD
};

struct lf_switch_data_change {
  int dc_type;				/* type of change */

  union {				/* specific data */

    struct {				/* for new or changed linecard */
      int slot;
      char *product_id;			/* product ID of new card */
      char *serial_no;			/* new serial number */
    } new_linecard;

    struct {				/* slot of missing card */
      int slot;
    } missing_linecard;

  } c;

  struct lf_switch_data_change *next;	/* next change */
};


/*
 * return codes from query routines
 */
enum {
  LF_QUERY_OK,
  LF_QUERY_RETRY,	/* error, but retry suggested */
  LF_QUERY_FATAL	/* error, don't bother with retry */
};

/*
 * number of times to retry querying the switch to allow for garbled
 * output from the switch.
 */
#define LF_SWITCH_QUERY_RETRY_COUNT 10

/*
 * prototypes
 */
int lf_query_switch(char *name, struct lf_enclosure **epp);
int lf_query_switch_x16_html(lf_enclosure_t *ep);
int lf_probe_switch_x16(lf_enclosure_t *ep);
int lf_query_switch_x32_html(lf_enclosure_t *ep);
int lf_probe_switch_x32(lf_enclosure_t *ep);
int lf_probe_switch_telnet_1(lf_enclosure_t *ep);
int lf_query_switch_telnet_1(lf_enclosure_t *ep);
FILE *lf_open_switch_html_stream(char *name, char *url);
int lf_note_missing_linecard(struct lf_enclosure *ep, int slot);
int lf_note_new_linecard(struct lf_enclosure *ep, int slot,
                         char *product_id, char *serial_no);
void lf_free_switch_change(struct lf_switch_data_change *cp);

#endif /* _lf_switch_h_ */
